#include "usbadc10.h"
#include <map>
#include <mutex>
#include <cstring>
#include <zf_log.h>
#include "urpc.h"
ZF_LOG_DEFINE_GLOBAL_OUTPUT_LEVEL;
static std::map<device_t, urpc_device_handle_t> impl_by_handle;
static std::mutex impl_by_handle_mutex;
static void push_data(uint8_t **where, const void *data, size_t size)
{
    memcpy(*where, data, size);
    *where += size;
}

#define GENERATE_PUSH(Type) \
static void push_##Type(uint8_t **where, Type value) { \
    push_data(where, &value, sizeof(value)); \
}
#define GENERATE_POP(Type) \
static Type pop_##Type(uint8_t **where) { \
    Type result; \
    memcpy(&result, *where, sizeof(result)); \
    *where += sizeof(result); \
    return (Type)result; \
}

GENERATE_PUSH(uint8_t)

GENERATE_POP(uint8_t)

GENERATE_POP(uint16_t)

GENERATE_POP(uint32_t)
device_t usbadc10_open_device(const char *uri)
{
    device_t handle;
    urpc_device_handle_t device;
    device = urpc_device_create(uri);
    if(device == NULL) {
        return device_undefined;
    }
    {
        std::lock_guard<std::mutex> lock(impl_by_handle_mutex);
        do
        {
            handle = rand();
        }
        while(impl_by_handle.count(handle) != 0);
        impl_by_handle[handle] = device;
    }
    return handle;
}

result_t usbadc10_libversion(char *lib_version)
{
    const char *lib_v = "1.0.1";
    strcpy(lib_version,lib_v);
    return result_ok;
}

result_t usbadc10_get_identity_information(device_t handle, usbadc10_get_identity_information_t *output)
{
    uint8_t *p;
    unsigned int i;
    uint8_t out_buffer[72];
    memset(out_buffer, 0, 72);
    urpc_device_handle_t device;
    if(handle < 0)
    {
        return result_error;
    }
    {
        std::lock_guard<std::mutex> lock(impl_by_handle_mutex);
        try
        {
            device = impl_by_handle.at(handle);
        }
        catch(const std::out_of_range &)
        {
            return result_error;
        }
    }
    urpc_result_t res;
    if((res = urpc_device_send_request(device, "ginf", NULL, 0, out_buffer, 72)) != urpc_result_ok)
    {
        result_t new_res;
        switch (res)
        {
        case urpc_result_ok:
             new_res = result_ok;
             break;
        case urpc_result_error:
             new_res = result_error;
             break;
        case urpc_result_value_error:
             new_res = result_value_error;
             break;
        case urpc_result_nodevice:
             new_res = result_nodevice;
             break;
        case urpc_result_timeout:
             new_res = result_timeout;
             break;
        default:
             new_res = res;
             break;
        }
        return new_res;
    }
    p = out_buffer;
    for(i=0; i<16; i++) output->Manufacturer[i] = pop_uint8_t(&p);
    for(i=0; i<16; i++) output->ProductName[i] = pop_uint8_t(&p);
    for(i=0; i<16; i++) output->ControllerName[i] = pop_uint8_t(&p);
    output->HardwareMajor = pop_uint8_t(&p);
    output->HardwareMinor = pop_uint8_t(&p);
    output->HardwareBugfix = pop_uint16_t(&p);
    output->BootloaderMajor = pop_uint8_t(&p);
    output->BootloaderMinor = pop_uint8_t(&p);
    output->BootloaderBugfix = pop_uint16_t(&p);
    output->FirmwareMajor = pop_uint8_t(&p);
    output->FirmwareMinor = pop_uint8_t(&p);
    output->FirmwareBugfix = pop_uint16_t(&p);
    output->SerialNumber = pop_uint32_t(&p);
    for(i=0; i<8; i++) output->Reserved[i] = pop_uint8_t(&p);
    return result_ok;
}

result_t usbadc10_get_conversion_raw(device_t handle, usbadc10_get_conversion_raw_t *output)
{
    uint8_t *p;
    unsigned int i;
    uint8_t out_buffer[20];
    memset(out_buffer, 0, 20);
    urpc_device_handle_t device;
    if(handle < 0)
    {
        return result_error;
    }
    {
        std::lock_guard<std::mutex> lock(impl_by_handle_mutex);
        try
        {
            device = impl_by_handle.at(handle);
        }
        catch(const std::out_of_range &)
        {
            return result_error;
        }
    }
    urpc_result_t res;
    if((res = urpc_device_send_request(device, "gdta", NULL, 0, out_buffer, 20)) != urpc_result_ok)
    {
        result_t new_res;
        switch (res)
        {
        case urpc_result_ok:
             new_res = result_ok;
             break;
        case urpc_result_error:
             new_res = result_error;
             break;
        case urpc_result_value_error:
             new_res = result_value_error;
             break;
        case urpc_result_nodevice:
             new_res = result_nodevice;
             break;
        case urpc_result_timeout:
             new_res = result_timeout;
             break;
        default:
             new_res = res;
             break;
        }
        return new_res;
    }
    p = out_buffer;
    for(i=0; i<10; i++) output->data[i] = pop_uint16_t(&p);
    return result_ok;
}

result_t usbadc10_get_conversion(device_t handle, usbadc10_get_conversion_t *output)
{
    uint8_t *p;
    unsigned int i;
    uint8_t out_buffer[20];
    memset(out_buffer, 0, 20);
    urpc_device_handle_t device;
    if(handle < 0)
    {
        return result_error;
    }
    {
        std::lock_guard<std::mutex> lock(impl_by_handle_mutex);
        try
        {
            device = impl_by_handle.at(handle);
        }
        catch(const std::out_of_range &)
        {
            return result_error;
        }
    }
    urpc_result_t res;
    if((res = urpc_device_send_request(device, "gcon", NULL, 0, out_buffer, 20)) != urpc_result_ok)
    {
        result_t new_res;
        switch (res)
        {
        case urpc_result_ok:
             new_res = result_ok;
             break;
        case urpc_result_error:
             new_res = result_error;
             break;
        case urpc_result_value_error:
             new_res = result_value_error;
             break;
        case urpc_result_nodevice:
             new_res = result_nodevice;
             break;
        case urpc_result_timeout:
             new_res = result_timeout;
             break;
        default:
             new_res = res;
             break;
        }
        return new_res;
    }
    p = out_buffer;
    for(i=0; i<10; i++) output->data[i] = pop_uint16_t(&p);
    return result_ok;
}

result_t usbadc10_get_calibration_settings(device_t handle, usbadc10_calibration_settings_t *output)
{
    uint8_t *p;
    unsigned int i;
    uint8_t out_buffer[4];
    memset(out_buffer, 0, 4);
    urpc_device_handle_t device;
    if(handle < 0)
    {
        return result_error;
    }
    {
        std::lock_guard<std::mutex> lock(impl_by_handle_mutex);
        try
        {
            device = impl_by_handle.at(handle);
        }
        catch(const std::out_of_range &)
        {
            return result_error;
        }
    }
    urpc_result_t res;
    if((res = urpc_device_send_request(device, "gcal", NULL, 0, out_buffer, 4)) != urpc_result_ok)
    {
        result_t new_res;
        switch (res)
        {
        case urpc_result_ok:
             new_res = result_ok;
             break;
        case urpc_result_error:
             new_res = result_error;
             break;
        case urpc_result_value_error:
             new_res = result_value_error;
             break;
        case urpc_result_nodevice:
             new_res = result_nodevice;
             break;
        case urpc_result_timeout:
             new_res = result_timeout;
             break;
        default:
             new_res = res;
             break;
        }
        return new_res;
    }
    p = out_buffer;
    for(i=0; i<4; i++) output->Reserved[i] = pop_uint8_t(&p);
    return result_ok;
}

result_t usbadc10_set_calibration_settings(device_t handle, usbadc10_calibration_settings_t *input)
{
    uint8_t *p;
    unsigned int i;
    uint8_t in_buffer[4];
    memset(in_buffer, 0, 4);
    urpc_device_handle_t device;
    if(handle < 0)
    {
        return result_error;
    }
    {
        std::lock_guard<std::mutex> lock(impl_by_handle_mutex);
        try
        {
            device = impl_by_handle.at(handle);
        }
        catch(const std::out_of_range &)
        {
            return result_error;
        }
    }
    p = in_buffer;
    for(i=0; i<4; i++) push_uint8_t(&p, input->Reserved[i]);
    urpc_result_t res;
    if((res = urpc_device_send_request(device, "scal", in_buffer, 4, NULL, 0)) != urpc_result_ok)
    {
        result_t new_res;
        switch (res)
        {
        case urpc_result_ok:
             new_res = result_ok;
             break;
        case urpc_result_error:
             new_res = result_error;
             break;
        case urpc_result_value_error:
             new_res = result_value_error;
             break;
        case urpc_result_nodevice:
             new_res = result_nodevice;
             break;
        case urpc_result_timeout:
             new_res = result_timeout;
             break;
        default:
             new_res = res;
             break;
        }
        return new_res;
    }
    return result_ok;
}

result_t usbadc10_close_device(device_t *handle_ptr)
{
    if(handle_ptr == NULL)
    {
        return result_error;
    }
    device_t handle = *handle_ptr;
    if(handle < 0)
    {
        return result_error;
    }
    urpc_device_handle_t device;
    {
        std::lock_guard<std::mutex> lock(impl_by_handle_mutex);
        try
        {
            device = impl_by_handle.at(handle);
        }
        catch(const std::out_of_range &)
        {
            return result_error;
        }
        impl_by_handle.erase(handle);
    }
    if(urpc_device_destroy(&device) != urpc_result_ok)
    {
        return result_error;
    }
    return result_ok;
}

